#ifndef __MT_APPLICATION_H__
#define __MT_APPLICATION_H__

#include "MTWindow.h"
#include "MTMenuBar.h"
#include "MTFile.h"
#include "MTAGLContext.h"

#include <fstream>

/***** MTApplication **********************************************************
 *	
 *	MTApplication is mostly an event handling class. It gets MacOS events,
 *	interprets them, performs any handling which can be locally performed, and
 *	dispatches the event to the correct window. It also handles creation of new
 *	windows, and sending object status messages.
 *	
 *	To use it:
 *		The menu handling functions and the window creation functions are the
 *		only functions which will probably need modification. Add menu
 *		selection handlers to the menu function, update the menu status
 *		function, and add any new windows to the window creation function. You
 *		will probably also want to change some of the library calls in the
 *		constructor/destructor.
 *	
 *****************************************************************************/

enum
{
	kIsDoubleClick =			1 << 0,
	
	kChangedWindowList =		1 << 0,
	kChangedWindowOrder =		1 << 1,
	kChangedMessageSupport =	1 << 2,
	kCompleteMenuUpdate =		0xFFFFFFFF
};

class MTApplication
{
	public:
		struct OpenResourceByIDRecord
		{
			UInt32	id;
			UInt8	canceled;
		};
		
		typedef UInt8 (*ModalDialogEventHandler)(DialogPtr theDialog, short item);
		
					MTApplication();
					~MTApplication();
		
		void		Init(void);
		
		void		Run(void);
		
		void		AttachWindow(MTWindow * theWindow);
		
		void		ObjectUpdated(UInt32 type, UInt32 object);
		void		ObjectDeleted(UInt32 type, UInt32 object);
		
		void		FreeWindow(WindowPtr theWindow);
		void		CloseAllOwnedBy(MTWindow * theWindow);
		
		void		SendMessageToTopWindow(UInt32 type, UInt32 data);
		UInt8		TopWindowSupportsMessage(UInt32 type, UInt32 data);
		
		void		UpdateMenuState(UInt32 needsUpdate);
		void		HandleUpdateEvent(EventRecord * theEvent);
		
		void		DoMoveableModalDialog(short id, short defaultItem, short cancelItem, ModalDialogEventHandler eventHandler, long refCon);
		
		static UInt8	DoOpenResourceByIDItemProc(DialogPtr theDialog, short item);
		
		UInt8		hasNavigation, hasAGL;
		UInt8		isDone;
	
	private:
		void		InitToolbox(void);
		
		void		InitAppleEventHandlers(void);
		void		DeInitAppleEventHandlers(void);
		
		void		HandleEvent(void);
		void		HandleMenuSelection(long selection);
		
		MTWindow *	LookupWindow(WindowPtr theWindow, UInt32 * id);
		
		MTWindow *	AttemptCreateWindowOfType(MTWindowType type);
		
		void		FreeAllWindows(void);
		
		UInt8		HasWindowOfType(MTWindowType type);
		UInt8		HasWindowOfType(MTWindowType type, MTWindow ** theWindow);
		
		MTWindow *	CreateWindowOfType(MTWindowType type);
		
		void		DoOpenFile(void);
		void		DoOpenFile(AEDescList * descList);
		void		DoOpenFile(FSSpec * theFileSpec);
		void		DoOpenResourceByID(void);
		
		static pascal Boolean	ModalFilterProc(DialogRef theDialog, EventRecord * theEvent, DialogItemIndex * itemHit);
		
		static pascal OSErr		HandleOpenApplicationEvent(const AppleEvent * theAppleEvent, AppleEvent * theReply, long refCon);
		static pascal OSErr		HandleOpenDocumentsEvent(const AppleEvent * theAppleEvent, AppleEvent * theReply, long refCon);
		static pascal OSErr		HandlePrintDocumentsEvent(const AppleEvent * theAppleEvent, AppleEvent * theReply, long refCon);
		static pascal OSErr		HandleQuitApplicationEvent(const AppleEvent * theAppleEvent, AppleEvent * theReply, long refCon);
		
		struct CMPODataEntry
		{
			UInt32	offset;
			UInt8	originalData;
			UInt8	patchData;
		};
		
		struct CMPODataRecord
		{
			std::string					targetFile;
			std::string					targetOS;
			std::vector <CMPODataEntry>	entries;
		};
		
		UInt32		ReadCMPO(CMPODataRecord * record);
		void		PrintCMPO(std::ostream & stream, CMPODataRecord * record);
		void		ApplyCMPO(void);
		void		RemoveCMPO(void);
		void		CreateCMPO(void);
		
		std::vector <MTWindow *>	windowList;
		MTMenuBar					menuBar;
		std::vector <MTFile *>		fileList;
		
		MTAGLContext				aglContext;
		
		ModalFilterUPP				modalFilterProcPtr;
		
		AEEventHandlerUPP			openApplicationHandlerUPP,
									openDocumentHandlerUPP,
									printDocumentHandlerUPP,
									quitApplicationHandlerUPP;
};

#if DEBUG_LOG
extern std::ofstream	debugLog;
#endif

extern MTApplication	* gTheApp;

#endif /* __MT_APPLICATION_H__ */